<?php

namespace App\Http\Controllers;

use App\Models\Message;
use App\Models\Product;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MessageController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        // Get all unique conversations
        $sentMessages = Message::where('sender_id', $user->id)->with(['receiver', 'product'])->get();
        $receivedMessages = Message::where('receiver_id', $user->id)->with(['sender', 'product'])->get();
        
        // Group by user
        $conversations = collect();
        
        foreach ($sentMessages as $message) {
            $key = $message->receiver_id;
            if (!$conversations->has($key)) {
                $conversations->put($key, [
                    'user' => $message->receiver,
                    'product' => $message->product,
                    'last_message' => $message,
                    'unread_count' => 0
                ]);
            }
        }
        
        foreach ($receivedMessages as $message) {
            $key = $message->sender_id;
            $unreadCount = $message->read_at ? 0 : 1;
            
            if ($conversations->has($key)) {
                $conversations[$key]['unread_count'] += $unreadCount;
                if ($message->created_at > $conversations[$key]['last_message']->created_at) {
                    $conversations[$key]['last_message'] = $message;
                }
            } else {
                $conversations->put($key, [
                    'user' => $message->sender,
                    'product' => $message->product,
                    'last_message' => $message,
                    'unread_count' => $unreadCount
                ]);
            }
        }
        
        // Sort by last message
        $conversations = $conversations->sortByDesc(function ($conv) {
            return $conv['last_message']->created_at;
        });

        return view('messages.index', compact('conversations'));
    }

    public function show(User $user, Request $request)
    {
        $productId = $request->get('product_id');
        
        $messages = Message::where(function ($query) use ($user, $productId) {
                $query->where('sender_id', Auth::id())
                      ->where('receiver_id', $user->id);
            })
            ->orWhere(function ($query) use ($user, $productId) {
                $query->where('sender_id', $user->id)
                      ->where('receiver_id', Auth::id());
            })
            ->when($productId, function ($query) use ($productId) {
                return $query->where('product_id', $productId);
            })
            ->with(['sender', 'receiver', 'product'])
            ->orderBy('created_at', 'asc')
            ->get();

        // Mark messages as read
        Message::where('sender_id', $user->id)
            ->where('receiver_id', Auth::id())
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        $conversations = $this->getConversations();
        
        return view('messages.show', compact('messages', 'user', 'conversations', 'productId'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'receiver_id' => 'required|exists:users,id',
            'product_id' => 'required|exists:products,id',
            'message' => 'required|string|max:1000',
        ]);

        // Prevent messaging yourself
        if ($validated['receiver_id'] == Auth::id()) {
            return back()->with('error', 'You cannot message yourself.');
        }

        $product = Product::findOrFail($validated['product_id']);
        
        // Prevent messaging if you're the owner
        if ($product->user_id == Auth::id()) {
            return back()->with('error', 'You cannot message yourself about your own product.');
        }

        Message::create([
            'sender_id' => Auth::id(),
            'receiver_id' => $validated['receiver_id'],
            'product_id' => $validated['product_id'],
            'message' => $validated['message'],
        ]);

        return back()->with('success', 'Message sent successfully!');
    }

    public function destroy(Message $message)
    {
        $this->authorize('delete', $message);
        
        $message->delete();
        
        return back()->with('success', 'Message deleted successfully!');
    }

    private function getConversations()
    {
        $user = Auth::user();
        
        $sentMessages = Message::where('sender_id', $user->id)->with(['receiver', 'product'])->get();
        $receivedMessages = Message::where('receiver_id', $user->id)->with(['sender', 'product'])->get();
        
        $conversations = collect();
        
        foreach ($sentMessages as $message) {
            $key = $message->receiver_id;
            if (!$conversations->has($key)) {
                $conversations->put($key, [
                    'user' => $message->receiver,
                    'product' => $message->product,
                    'last_message' => $message,
                    'unread_count' => 0
                ]);
            }
        }
        
        foreach ($receivedMessages as $message) {
            $key = $message->sender_id;
            $unreadCount = $message->read_at ? 0 : 1;
            
            if ($conversations->has($key)) {
                $conversations[$key]['unread_count'] += $unreadCount;
                if ($message->created_at > $conversations[$key]['last_message']->created_at) {
                    $conversations[$key]['last_message'] = $message;
                }
            } else {
                $conversations->put($key, [
                    'user' => $message->sender,
                    'product' => $message->product,
                    'last_message' => $message,
                    'unread_count' => $unreadCount
                ]);
            }
        }
        
        return $conversations->sortByDesc(function ($conv) {
            return $conv['last_message']->created_at;
        });
    }
}
