<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\User;
use App\Models\Message;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class AdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('admin');
    }

    public function dashboard()
    {
        $totalUsers = User::count();
        $totalProducts = Product::count();
        $activeProducts = Product::where('status', 'active')->count();
        $soldProducts = Product::where('status', 'sold')->count();
        $totalMessages = Message::count();

        return view('admin.dashboard', compact(
            'totalUsers',
            'totalProducts',
            'activeProducts',
            'soldProducts',
            'totalMessages'
        ));
    }

    public function users()
    {
        $users = User::latest()->paginate(20);
        return view('admin.users', compact('users'));
    }

    public function products()
    {
        $products = Product::with(['user', 'category'])->latest()->paginate(20);
        return view('admin.products', compact('products'));
    }

    public function deleteProduct(Product $product)
    {
        if ($product->images) {
            foreach ($product->images as $image) {
                Storage::disk('public')->delete($image);
            }
        }

        $product->delete();

        return back()->with('success', 'Product deleted successfully!');
    }

    public function deleteUser(User $user)
    {
        // Delete all products owned by the user
        foreach ($user->products as $product) {
            if ($product->images) {
                foreach ($product->images as $image) {
                    Storage::disk('public')->delete($image);
                }
            }
            $product->delete();
        }

        // Delete all messages sent or received by the user
        Message::where('sender_id', $user->id)->orWhere('receiver_id', $user->id)->delete();

        // Delete avatar
        if ($user->avatar) {
            Storage::disk('public')->delete($user->avatar);
        }

        $user->delete();

        return back()->with('success', 'User and all related data deleted successfully!');
    }
}
